#include "ship.h"
#include <iostream>

using namespace std;

Ship::Ship() {}

Ship::Ship(CFDSolver* cfd, COLOR color, position2df start, int rotDegrees, IVideoDriver* driver, int player, FMOD::System* fmodSystem) :
	Entity(cfd, color, start, rotDegrees, driver, player, fmodSystem),
	m_missile(new Missile(cfd, color, start, rotDegrees, driver, player, fmodSystem)),
	m_mine(new Mine(cfd, color, start, rotDegrees, driver, player, fmodSystem)),
	m_health(MAX_SHIP_HEALTH),
	m_mineReloadCounter(0),
	m_missileReloadCounter(0),
	m_exploded(false),
	m_explodeTime(100),
	m_lastDamage(0.0) {

	stringc filename, shieldFilename;
	switch (color) {
	case COLOR_RED:
		filename = "gfx/fighter_red";
		shieldFilename = "gfx/shield_red.png";
		break;
	case COLOR_GREEN:
		filename = "gfx/fighter_green";
		shieldFilename = "gfx/shield_green.png";
		break;
	case COLOR_BLUE:
		filename = "gfx/fighter_blue";
		shieldFilename = "gfx/shield_blue.png";
		break;
	case COLOR_BLACK:
		filename = "gfx/fighter_black";
		shieldFilename = "gfx/shield_black.png";
		break;
	default: break;
	}
	
	for (int i = 0; i < 72; i++) {
		stringc number = i;
		stringc fullName = filename + "_" + number + ".png";
		m_textureArray[i] = driver->getTexture(fullName.c_str());
		stringc thrustName = filename + "_thrust_" + number + ".png";
		m_textureThrustArray[i] = driver->getTexture(thrustName.c_str());
	}
	m_texture = m_textureArray[m_rotation/5];
	m_shield = driver->getTexture(shieldFilename.c_str());

	m_displayRect = rect<s32>(0, 0, SHIP_SPRITE_WIDTH, SHIP_SPRITE_HEIGHT);
	m_shieldDisplayRect = core::rect<s32>(
			0, 0, SHIELD_SPRITE_WIDTH, SHIELD_SPRITE_HEIGHT);

	m_keymap = new EKEY_CODE[5];
	if (player == 1) {
		m_keymap[0] = KEY_KEY_W;
		m_keymap[1] = KEY_KEY_A;
		m_keymap[2] = KEY_KEY_S;
		m_keymap[3] = KEY_KEY_D;
		m_keymap[4] = KEY_LCONTROL;
	} else {
		m_keymap[0] = KEY_UP;
		m_keymap[1] = KEY_LEFT;
		m_keymap[2] = KEY_DOWN;
		m_keymap[3] = KEY_RIGHT;
		m_keymap[4] = KEY_RCONTROL;
	}

	float centerX = SHIP_SPRITE_WIDTH / 2.0;
	float centerY = SHIP_SPRITE_HEIGHT / 2.0;
	float radius = MAX(SHIP_SPRITE_WIDTH / 2.0,SHIP_SPRITE_HEIGHT / 2.0);

	for(int x=0; x<SHIP_SPRITE_WIDTH; x++) {
		for(int y=0; y<SHIP_SPRITE_HEIGHT; y++) {
			if(sqrtf((x-centerX)*(x-centerX) + (y-centerY)*(y-centerY)) <= radius) { // if distance < radius
				int cfdX = ((int) (x * GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
				int cfdY = ((int) (y * GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;
				m_gridCells.insert(CoordPair(cfdX,cfdY));
			}
		}
	}
	//Initialize sounds
	errcheck(m_fmodSystem->createSound("sfx/ship_thrust.wav", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE,0,&m_thrustSound));
	errcheck(m_fmodSystem->createSound("sfx/ship_explode.wav", FMOD_LOOP_OFF|FMOD_2D|FMOD_SOFTWARE,0,&m_explodeSound));
	errcheck(m_fmodSystem->createSound("sfx/alarm_1.wav", FMOD_LOOP_OFF|FMOD_2D|FMOD_SOFTWARE,0,&m_alarm1));
	errcheck(m_fmodSystem->createSound("sfx/alarm_2.wav", FMOD_LOOP_OFF|FMOD_2D|FMOD_SOFTWARE,0,&m_alarm2));
	errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_thrustSound, true, &m_thrustChannel));
}

Ship::~Ship() {
	delete m_missile;
	delete m_mine;
}

void Ship::takeDamage(float dmg, int dt) {
	if (m_health - dmg * dt < FLASH_HP_FAST * MAX_SHIP_HEALTH
			&& m_health > FLASH_HP_FAST * MAX_SHIP_HEALTH)
		errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_alarm2, false, NULL));
	else if (m_health - dmg * dt < FLASH_HP_SLOW * MAX_SHIP_HEALTH
			&& m_health > FLASH_HP_SLOW * MAX_SHIP_HEALTH)
		errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_alarm1, false, NULL));

	m_health -= dmg * dt;
	m_lastDamage = dmg * dt;

	if (m_health <= 0.0 && ! m_exploded) {
		m_exploded = true;
		detonate();
	}
}

void Ship::detonate() {
	int cfdX = ((int) ((m_position.X + SHIP_SPRITE_WIDTH / 2)
			* GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
	int cfdY = ((int) ((m_position.Y + SHIP_SPRITE_HEIGHT / 2)
			* GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;
	m_cfd->addDensityAt(SHIP_DENSITY_ADD, cfdX, cfdY);
	for (int i = -1; i <= 1; i++) {
		for (int j = -1; j <= 1; j++) {
			m_cfd->addUForce(i * 10000.0, cfdX+i, cfdY+j);
			m_cfd->addVForce(j * 10000.0, cfdX+i, cfdY+j);
		}
	}
	errcheck(m_thrustChannel->setPaused(true));
	errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_explodeSound, false, NULL));
}

void Ship::update(int dt, bool* keys, CFDSolver* cfd) {
	// get point for CFD effects
	int cfdX = ((int) ((m_position.X + SHIP_SPRITE_WIDTH / 2)
			* GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
	int cfdY = ((int) ((m_position.Y + SHIP_SPRITE_HEIGHT / 2)
			* GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;
	int cfdXOrigin = ((int) (m_position.X * GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
	int cfdYOrigin = ((int) (m_position.Y * GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;

	if (m_exploded && m_explodeTime > 0) {
		m_explodeTime -= dt;
		//Explosion code
		for (int i = -1; i <= 1; i++) {
			for (int j = -1; j <= 1; j++) {
				m_cfd->addUForce(i * 10000.0, cfdX+i, cfdY+j);
				m_cfd->addVForce(j * 10000.0, cfdX+i, cfdY+j);
			}
		}
	}

	// timestep
	if (dt) {
		// update health
		// float damage = cfd->getDensityAt(cfdX, cfdY);
		float damage = 0.0;
		for(CoordSet::iterator iter = m_gridCells.begin(); iter != m_gridCells.end(); iter++) {
			CoordPair pos = *iter;
			int x = (cfdXOrigin + pos.first) % GRID_SIZE;
			int y = (cfdYOrigin + pos.second) % GRID_SIZE;
			damage += cfd->getDensityAt(x, y);
		}
		m_lastDamage = 0.0;
		takeDamage(DAMAGE_FACTOR * damage, dt);

		// tick reload counters
		m_mineReloadCounter += dt;
		m_missileReloadCounter += dt;

		// update munitions
		if (m_mine->m_active)
			m_mine->update(dt, keys, cfd);
		if (m_missile->m_active)
			m_missile->update(dt, keys, cfd);

		// update position
		m_position.X += m_velocity.X * (float)dt;
		m_position.Y += m_velocity.Y * (float)dt;

	
		// damp velocity
		m_velocity *= m_velocityDamp;
	
		// torus wrap position
		if(m_position.X > SCREEN_WIDTH)
			m_position.X -= SCREEN_WIDTH;
		else if(m_position.X < 0)
			m_position.X += SCREEN_WIDTH;
		if(m_position.Y > SCREEN_HEIGHT)
			m_position.Y -= SCREEN_HEIGHT;
		else if(m_position.Y < 0)
			m_position.Y += SCREEN_HEIGHT;
	
		// push fluid out of the way
		cfd->addUForce(FLUID_PUSH_FACTOR*m_velocity.X, cfdX, cfdY);
		cfd->addVForce(FLUID_PUSH_FACTOR*m_velocity.Y, cfdX, cfdY);
	}
	
	if (m_exploded)
		return;

	// respond to keyboard input
	if (keys[m_keymap[1]]) { //A
		rotate(-5);
	}
	if (keys[m_keymap[2]]) { //S
		if (m_mine->m_active) {
			m_mine->detonate();
			m_mineReloadCounter = 0;
		} else if (m_mineReloadCounter > m_reloadTime) {
			position2df position;
			position.X = m_position.X + (SHIP_SPRITE_WIDTH / 2.0)
				- (MINE_SPRITE_WIDTH / 2.0);
			position.Y = m_position.Y + (SHIP_SPRITE_HEIGHT / 2.0)
				- (MINE_SPRITE_HEIGHT / 2.0);
			m_mine->activate(m_rotation, m_velocity, position);
		}
	}
	if (keys[m_keymap[3]]) { //D
		rotate(5);
	}
	if (keys[m_keymap[4]]) { //Space
		if (m_missile->m_active) {
			m_missile->detonate();
			m_missileReloadCounter = 0;
		} else if (m_missileReloadCounter > m_reloadTime) {
			m_missile->activate(m_rotation, m_velocity, m_position);
		}
	}
	if (keys[m_keymap[0]]) { //W
		applyThrust(cfdX,cfdY,cfd);
		m_texture = m_textureThrustArray[m_rotation/5];
		errcheck(m_thrustChannel->setPaused(false));
	} else {
		m_texture = m_textureArray[m_rotation/5];
		errcheck(m_thrustChannel->setPaused(true));
	}
}
