#include "ship.h"
#include "missile.h"
#include <iostream>

Missile::Missile() {}

Missile::Missile(CFDSolver* cfd, COLOR color, position2df start, int rotDegrees, IVideoDriver* driver, int player, FMOD::System* fmodSystem) :
	Entity(cfd, color, start, rotDegrees, driver, player, fmodSystem),
	m_active(false),
	m_timePassed(0) {

	stringc filename;
	switch (color) {
	case COLOR_RED:
		filename = "gfx/missile_red";
		break;
	case COLOR_GREEN:
		filename = "gfx/missile_green";
		break;
	case COLOR_BLUE:
		filename = "gfx/missile_blue";
		break;
	case COLOR_BLACK:
		filename = "gfx/missile_black";
		break;
	}
	
	for (int i = 0; i < 72; i++) {
		stringc number = i;
		stringc fullName = filename + "_" + number + ".png";
		m_textureArray[i] = driver->getTexture(fullName.c_str());
		stringc thrustName = filename + "_thrust_" + number + ".png";
		m_textureThrustArray[i] = driver->getTexture(thrustName.c_str());
	}
	m_texture = m_textureArray[0];

	m_displayRect = rect<s32>(0, 0, MISSILE_SPRITE_WIDTH, MISSILE_SPRITE_HEIGHT);
	//Initialize sounds
	errcheck(m_fmodSystem->createSound("sfx/missile_thrust.wav", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE,0,&m_thrustSound));
	errcheck(m_fmodSystem->createSound("sfx/missile_explode.mp3", FMOD_LOOP_OFF|FMOD_2D|FMOD_SOFTWARE,0,&m_explodeSound));
	errcheck(m_fmodSystem->createSound("sfx/missile_launch.wav", FMOD_LOOP_OFF|FMOD_2D|FMOD_SOFTWARE,0,&m_launchSound));
	errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_thrustSound, true, &m_thrustChannel));
}

Missile::~Missile() {
}

void Missile::update(int dt, bool* keys, CFDSolver* cfd) {
	m_timePassed += dt;
	// get point for CFD effects
	int cfdX = ((int) ((m_position.X + MISSILE_SPRITE_WIDTH / 2)
			* GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
	int cfdY = ((int) ((m_position.Y + MISSILE_SPRITE_HEIGHT / 2)
			* GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;
	

	//Change the texture to non thrusting - 
	//applyThrust will change it if it gets called
	m_texture = m_textureArray[m_rotation/5];


	if (m_timePassed > m_beforeThrust && m_timePassed < m_thrustStops) {
		errcheck(m_thrustChannel->setPaused(false));
		applyThrust(cfdX, cfdY, cfd);
	} else if (m_timePassed > m_thrustStops) {
		errcheck(m_thrustChannel->setPaused(true));
	}
	if (m_timePassed > m_autoDetonate)
		detonate();
	// timestep
	if (dt) {
		// update position
		m_position.X += m_velocity.X * (float)dt;
		m_position.Y += m_velocity.Y * (float)dt;
	
		// damp velocity
		m_velocity *= m_velocityDamp;
	
		// torus wrap position
		if(m_position.X > SCREEN_WIDTH)
			m_position.X -= SCREEN_WIDTH;
		else if(m_position.X < 0)
			m_position.X += SCREEN_WIDTH;
		if(m_position.Y > SCREEN_HEIGHT)
			m_position.Y -= SCREEN_HEIGHT;
		else if(m_position.Y < 0)
			m_position.Y += SCREEN_HEIGHT;
	
		// push fluid out of the way
		cfd->addUForce(FLUID_PUSH_FACTOR*m_velocity.X, cfdX, cfdY);
		cfd->addVForce(FLUID_PUSH_FACTOR*m_velocity.Y, cfdX, cfdY);
	}
}

void Missile::activate(int rotDegrees, vector2df velocity, position2df position) {
	errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_launchSound, false, NULL));
	m_active = true;
	m_position = position;
	m_rotation = rotDegrees;
	m_texture = m_textureArray[rotDegrees/5];
	vector2df tmp(sin(TORADIANS(rotDegrees)),-1.0*cos(TORADIANS(rotDegrees)));
	m_velocity = velocity + tmp * 0.25;
	m_timePassed = 0;
}

void Missile::detonate() {
	if (m_timePassed < m_beforeDetonate)
		return;
	m_active = false;
	m_timePassed = 0;
	int cfdX = ((int) ((m_position.X + MISSILE_SPRITE_WIDTH / 2)
			* GRID_SIZE / SCREEN_WIDTH)) % GRID_SIZE;
	int cfdY = ((int) ((m_position.Y + MISSILE_SPRITE_HEIGHT / 2)
			* GRID_SIZE / SCREEN_HEIGHT)) % GRID_SIZE;
	m_cfd->addDensityAt(MISSILE_DENSITY_ADD, cfdX, cfdY);
	errcheck(m_thrustChannel->setPaused(true));
	errcheck(m_fmodSystem->playSound(FMOD_CHANNEL_FREE, m_explodeSound, false, NULL));
}
