#include <irrlicht.h>
#include <iostream>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <fmod.hpp>
#include <fmod_errors.h>
#include "entity.h"
#include "ship.h"
#include "missile.h"
#include "mine.h"
#include "CFDSolver.h"

using namespace irr;
using namespace std;

#define DIFFUSION 0.0001
#define VISCOSITY 0.2

#pragma comment(lib, "Irrlicht.lib")
IrrlichtDevice* device;
Ship* player1, * player2;
CFDSolver* cfd;
video::IVideoDriver* driver;
gui::IGUIFont* bigfont;
int winner;
int player1health, player2health;
unsigned int timeNow, lastTime;
int flashCount = 0; // health bar flash ticker in ms
COLOR player1Color = COLOR_NONE, player2Color = COLOR_NONE;
video::SColor player1IrrColor, player2IrrColor;
video::ITexture* whiteShield;
bool colorSelectUp = true;
bool controlsUp = false;
bool loaded = false;
FMOD::System* fmodSystem;
FMOD::Sound* bgm;
FMOD::Channel* bgmChannel;

void drawShip(Ship* player, video::SColor playerColor);
void drawMissile(Missile* missile);
void drawMine(Mine* mine);
void drawWinner(int winner, int hp);
void newGame();
bool drawHealthBar(Ship* player);
void chooseColor(char color);
//Everything that should have been put in a header file ends here.

class MyEventReceiver : public IEventReceiver {
public:
	MyEventReceiver() {
		keys = new bool[KEY_KEY_CODES_COUNT];
		memset(keys, 0, sizeof(bool)*KEY_KEY_CODES_COUNT);
	}
	~MyEventReceiver() {
		delete[] keys; //This might be causing an occasional double free on exit.
	}
	bool OnEvent(const SEvent& event) {
		if(event.EventType == irr::EET_KEY_INPUT_EVENT) {
			keys[event.KeyInput.Key] = event.KeyInput.PressedDown;
			switch (event.KeyInput.Key) {
				case KEY_KEY_1:
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/Minor_Disturbance.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
					return true;
				case KEY_KEY_2:
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/Sax_Quartet.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
					return true;
				case KEY_KEY_3:
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/Marcs_Fugue.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
					return true;
				case KEY_KEY_4:
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/Toccata_and_Fugue.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
					return true;
				case KEY_KEY_0:
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
					}
					return true;
				case KEY_ESCAPE:
					device->closeDevice();
					return true;
				case KEY_KEY_N:
					if (winner || controlsUp) {
						newGame();
						return true;
					}
					break;
				case KEY_KEY_R: chooseColor('r'); break;
				case KEY_KEY_G: chooseColor('g'); break;
				case KEY_KEY_B: chooseColor('b'); break;
				default: return false;
			}
		}
		return false;
	}
	bool* keys;
};


int main(int argc, char *argv[]) {
	// make irrlicht stop spamming console when it loads textures
	close(1);
	int devnull = open("/dev/null", O_WRONLY);
	dup(devnull);

	// let user select driver type...or not
	video::E_DRIVER_TYPE driverType;

	/*printf("Please select the driver you want for this example:\n"\
		" (a) Direct3D 9.0c\n (b) Direct3D 8.1\n (c) OpenGL 1.5\n"\
		" (d) Software Renderer\n (e) Burning's Software Renderer\n"\
		" (f) NullDevice\n (otherKey) exit\n\n");
		
	char i;
	std::cin >> i;

	switch(i) {
		case 'a': driverType = video::EDT_DIRECT3D9;break;
		case 'b': driverType = video::EDT_DIRECT3D8;break;
		case 'c': driverType = video::EDT_OPENGL;   break;
		case 'd': driverType = video::EDT_SOFTWARE; break;
		case 'e': driverType = video::EDT_BURNINGSVIDEO;break;
		case 'f': driverType = video::EDT_NULL;     break;
		default: return 0;
	}*/
	driverType = video::EDT_OPENGL;

	// create device
	device = createDevice(driverType,
		core::dimension2d<s32>(SCREEN_WIDTH, SCREEN_HEIGHT));

	if (device == 0)
		return 1; // could not create selected driver.

	device->setWindowCaption(L"Hydromancy - SmokeDuel");

	driver = device->getVideoDriver();

	MyEventReceiver receiver;
	device->setEventReceiver(&receiver);

	//Set up fmod stuff
	errcheck(FMOD::System_Create(&fmodSystem));
	errcheck(fmodSystem->setSpeakerMode(FMOD_SPEAKERMODE_STEREO));
	errcheck(fmodSystem->init(100, FMOD_INIT_NORMAL, 0));

	video::ITexture* background = driver->getTexture("gfx/stars.jpg");
	video::ITexture* splashScreen = driver->getTexture("gfx/splash.png");
	whiteShield = driver->getTexture("gfx/shield_white.png");

	gui::IGUIFont* font = device->getGUIEnvironment()->getFont("gfx/fontlucida.png");
	bigfont = device->getGUIEnvironment()->getFont("gfx/fontbig.png");
	stringw p1 = L"Player 1: ";
	stringw p2 = L"Player 2: ";

	while(device->run() && driver) {
		if (device->isWindowActive()) {
			lastTime = timeNow;
			timeNow = device->getTimer()->getTime();
			int dt = timeNow-lastTime; // TODO: ?CFD? slowdown at mine peak (70% shields ish on mine)
			flashCount += dt;

			driver->beginScene(true, true, video::SColor(0,0,0,0));


			if(colorSelectUp) {
				// draw splash screen
				driver->draw2DImage(splashScreen,
					core::rect<s32>(0,0, SCREEN_WIDTH, SCREEN_HEIGHT), //Dest
					core::rect<s32>(0,0,800,600)); //Src
				if(!loaded) { // do not combine this with !loaded if statement below (endscene in between)
					stringw loading = L"Loading...";
					bigfont->draw(loading.c_str(),
						core::rect<s32>(0, 0, SCREEN_WIDTH, 240),
						video::SColor(255,255,255,255), true, true);
				} else {
					stringw colorSelect, colorSelect2;
					if(player1Color == COLOR_NONE) {
						colorSelect = L"Player 1, select a color:";
						colorSelect2 = L"(R)ed  (G)reen  (B)lue";
					} else {
						colorSelect = L"Player 2, select a color: ";
						colorSelect2 = (player1Color == COLOR_RED ? L"" : L"(R)ed  ");
						colorSelect2 += (player1Color == COLOR_GREEN ? L"" : L"(G)reen  ");
						colorSelect2 += (player1Color == COLOR_BLUE ? L"" : L"(B)lue");
					}
					bigfont->draw(colorSelect.c_str(),
						core::rect<s32>(0, 60, SCREEN_WIDTH, 300),
						video::SColor(255,255,0,0), true, true);
					bigfont->draw(colorSelect2.c_str(),
						core::rect<s32>(0, 120, SCREEN_WIDTH, 300 + 40),
						video::SColor(255,255,0,0), true, true);
				}

				driver->endScene();

				if(!loaded) {
					loaded = true;
					delete(new Ship(cfd, COLOR_RED, core::position2df(SCREEN_WIDTH/4.0-30.0, SCREEN_HEIGHT/2.0-10.0), 0, driver, 1, fmodSystem));
					delete(new Ship(cfd, COLOR_GREEN, core::position2df(SCREEN_WIDTH/4.0-30.0, SCREEN_HEIGHT/2.0-10.0), 0, driver, 1, fmodSystem));
					delete(new Ship(cfd, COLOR_BLUE, core::position2df(SCREEN_WIDTH/4.0-30.0, SCREEN_HEIGHT/2.0-10.0), 0, driver, 1, fmodSystem));
				}
				continue;
			}
			// else if controls page up
			if(controlsUp) {
				// Say loading, draw conrols here, n for new game
				int controlsCount = 20;
				stringw controls[] = {L"General Controls:", L"N - New Game", L"Escape - Quit", L"1-4 - Music selection", L"(in order of epicness)", L"0 - No Music", L"", L"Player 1:", L"W - Thrust", L"A - Turn Left", L"D - Turn Right", L"S - Drop/Detonate Mine", L"Left Control - Fire/Detonate Missile", L"", L"Player 2: ", L"Up Arrow - Thrust", L"Left Arrow - Turn Left", L"Right Arrow - Turn Right", L"Down Arrow - Drop/Detonate Mine", L"Right Control - Fire/Detonate Missile"};
				for(int i=0; i<controlsCount; i++) {
					bigfont->draw(controls[i].c_str(),
						core::rect<s32>(0, 0, SCREEN_WIDTH, 100 + i*45),
						video::SColor(255,255,255,255), true, true);
				}
				stringw continuestr = L"Press N to start the game";
				bigfont->draw(continuestr.c_str(),
					core::rect<s32>(0, 0, SCREEN_WIDTH, 100 + controlsCount*45 + 100),
					video::SColor(255,255,0,0), true, true);

				driver->endScene();
				continue;
			}
			// else draw game

			// draw background
			driver->draw2DImage(background,
				core::rect<s32>(0,0, SCREEN_WIDTH, SCREEN_HEIGHT), //Dest
				core::rect<s32>(0,0,1280,1024)); //Src

			if (winner == 0) {
				//Find a winner
				if (player1->m_exploded) {
					winner = 2;
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/French_Quartet_2.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
				} else if (player2->m_exploded) {
					winner = 1;
					if (bgmChannel) {
						errcheck(bgmChannel->setPaused(true));
						errcheck(fmodSystem->createStream("bgm/Ave_Verum.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
						errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
					}
				}

				//Record health for winner screen
				player1health = (int) player1->m_health;
				player2health = (int) player2->m_health;
				

			}

			// update ships
			player1->update(dt, receiver.keys, cfd);
			player2->update(dt, receiver.keys, cfd);

			//draw ships
			if (! player1->m_exploded) {
				drawShip(player1,player1IrrColor);
			}
				if(player1->m_missile->m_active)
					drawMissile(player1->m_missile);
				if(player1->m_mine->m_active)
					drawMine(player1->m_mine);

			if (! player2->m_exploded) {
				drawShip(player2,player2IrrColor);
			}
				if(player2->m_missile->m_active)
					drawMissile(player2->m_missile);
				if(player2->m_mine->m_active)
					drawMine(player2->m_mine);

			cfd->runTimeStep(((float)dt)/250.0);
			//draw CFD rects
			for (int i = 1; i <= GRID_SIZE; i++) {
				for (int j = 1; j <= GRID_SIZE; j++) {
					float density = cfd->getDensityAt(i, j);// * 0.5;
					if (density < 1.0/255.0) continue;
					if (density > 1.0) density = 1.0;
					int x = (i-1)*SCREEN_WIDTH/GRID_SIZE;
					int xPlus1 = i*SCREEN_WIDTH/GRID_SIZE;
					int y = (j-1)*SCREEN_HEIGHT/GRID_SIZE;
					int yPlus1 = j*SCREEN_HEIGHT/GRID_SIZE;
					driver->draw2DRectangle(
						video::SColor((int)(255*density), 255, 255, 255),
						core::rect<s32>(x, y,xPlus1, yPlus1));
				}
			}
			
			float healthBarSizeFactor = 1.5;
			// draw p1 status
			int p1health = (int)player1->m_health;
			if (p1health < 0)
				p1health = 0;
			// draw text status
			font->draw((p1 + p1health + L"% shields").c_str(), 
				core::rect<s32>(5,SCREEN_HEIGHT-40,155,SCREEN_HEIGHT-20),
				player1IrrColor);
			if (drawHealthBar(player1)) {
				// draw health bar
				driver->draw2DRectangle(player1IrrColor,core::rect<s32>(5,SCREEN_HEIGHT-20,
							5+(int)(p1health * healthBarSizeFactor),SCREEN_HEIGHT-10));
				// draw health bar background
				video::SColor bg = player1IrrColor; bg.setAlpha(150);
				driver->draw2DRectangle(bg,core::rect<s32>(5,SCREEN_HEIGHT-20,
							5+(int)(MAX_SHIP_HEALTH * healthBarSizeFactor),SCREEN_HEIGHT-10));
			}

			// draw p2 status
			int p2drawOffset = 160;
			int p2health = (int)player2->m_health;
			if (p2health < 0)
				p2health = 0;
			font->draw((p2 + p2health + L"% shields").c_str(), 
				core::rect<s32>(SCREEN_WIDTH-p2drawOffset,SCREEN_HEIGHT-40,155,SCREEN_HEIGHT-20),
				player2IrrColor);
			if (drawHealthBar(player2)) {
				// draw health bar
				driver->draw2DRectangle(player2IrrColor,core::rect<s32>(SCREEN_WIDTH-p2drawOffset,
							SCREEN_HEIGHT-20,
							SCREEN_WIDTH-p2drawOffset + (int)(p2health * healthBarSizeFactor),
							SCREEN_HEIGHT-10));
				video::SColor bg = player2IrrColor; bg.setAlpha(150);
				driver->draw2DRectangle(bg,core::rect<s32>(SCREEN_WIDTH-p2drawOffset,
							SCREEN_HEIGHT-20,
							SCREEN_WIDTH-p2drawOffset + (int)(MAX_SHIP_HEALTH * healthBarSizeFactor),
							SCREEN_HEIGHT-10));
			}

			if (winner == 1)
				drawWinner(winner, player1health);
			else if (winner == 2)
				drawWinner(winner, player2health);
			driver->endScene();
		}
	}

	delete cfd;
	delete player1;
	delete player2;
	device->drop();

	return 0;
}

void drawWinner(int winner, int hp) {
	video::SColor color = (winner == 1) ?
		player1IrrColor : player2IrrColor;
	stringw victory = L"Player ";
	victory += winner;
	victory += L" Wins";
	stringw victory2 = L"with ";
	victory2 += hp;
	victory2 += L"% shields left!";
	stringw newgamenote = L"Press 'N' for a new game";
	bigfont->draw(victory.c_str(),
		core::rect<s32>(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT),
		color,
		true, true);
	bigfont->draw(victory2.c_str(),
		core::rect<s32>(0, 50, SCREEN_WIDTH, SCREEN_HEIGHT),
		color,
		true, true);
	bigfont->draw(newgamenote.c_str(),
		core::rect<s32>(0, 200, SCREEN_WIDTH, SCREEN_HEIGHT),
		color,
		true, true);
}


void drawShip(Ship* player, video::SColor playerColor) {
	int playerx = (int) (player->m_position.X);
	int playery = (int) (player->m_position.Y);

	// draw normally
	driver->draw2DImage(player->m_texture,
		core::rect<s32>(playerx,playery,
		playerx + SHIP_SPRITE_WIDTH, playery + SHIP_SPRITE_HEIGHT),
		player->m_displayRect,
		0, 0, true);

	int shieldRadius = (int)MAX(MAX_SHIELD_RADIUS-(MAX_SHIP_HEALTH - player->m_health)/3.0, MIN_SHIELD_RADIUS);

	//Draw emergency (white) shield
	int whiteAlpha;
	video::SColor whiteCorners[4];
	if (player->m_health < FLASH_HP_SLOW * MAX_SHIP_HEALTH &&
			player->m_health > FLASH_HP_FAST * MAX_SHIP_HEALTH) {
		whiteAlpha = flashCount% (2 * FLASH_TIME_SLOW);
		whiteAlpha = MIN(whiteAlpha, 2*FLASH_TIME_SLOW-whiteAlpha);
		whiteAlpha = (int)(whiteAlpha * 200.0/FLASH_TIME_SLOW);
				
		whiteCorners[0] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[1] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[2] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[3] = video::SColor(whiteAlpha,255,255,255);
	} else if (player->m_health < FLASH_HP_FAST * MAX_SHIP_HEALTH) {
		whiteAlpha = flashCount% (FLASH_TIME_FAST);
		whiteAlpha = MIN(whiteAlpha, FLASH_TIME_FAST-whiteAlpha);
		whiteAlpha = (int)(whiteAlpha * 255.0/FLASH_TIME_FAST);
		//whiteAlpha *= 150.0/FLASH_TIME_FAST;
				
		whiteCorners[0] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[1] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[2] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[3] = video::SColor(whiteAlpha,255,255,255);
		/*whiteCorners[0] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[1] = video::SColor(150 - whiteAlpha,255,255,255);
		whiteCorners[2] = video::SColor(whiteAlpha,255,255,255);
		whiteCorners[3] = video::SColor(150 - whiteAlpha,255,255,255);
		*/
	}
	if (player->m_health < FLASH_HP_SLOW * MAX_SHIP_HEALTH) {
		driver->draw2DImage(whiteShield,
		core::rect<s32>(playerx+(SHIP_SPRITE_WIDTH/2)-(shieldRadius),
			playery+(SHIP_SPRITE_HEIGHT/2)-(shieldRadius),
			playerx+(SHIP_SPRITE_WIDTH/2)+(shieldRadius),
			playery+(SHIP_SPRITE_HEIGHT/2)+(shieldRadius)),
		player->m_shieldDisplayRect,
		0, whiteCorners, true);
	}
	
	//Draw normal shields
	int shieldAlpha = MIN((int) (player->m_lastDamage * 255.0 * 3.0), 175);
	if (player->m_lastDamage > 0.005)
		shieldAlpha = MAX(shieldAlpha, 40);
	video::SColor corners[4];
	corners[0] = video::SColor(shieldAlpha,255,255,255);
	corners[1] = video::SColor(shieldAlpha,255,255,255);
	corners[2] = video::SColor(shieldAlpha,255,255,255);
	corners[3] = video::SColor(shieldAlpha,255,255,255);
	driver->draw2DImage(player->m_shield,
		core::rect<s32>(playerx+(SHIP_SPRITE_WIDTH/2)-(shieldRadius),
			playery+(SHIP_SPRITE_HEIGHT/2)-(shieldRadius),
			playerx+(SHIP_SPRITE_WIDTH/2)+(shieldRadius),
			playery+(SHIP_SPRITE_HEIGHT/2)+(shieldRadius)),
		player->m_shieldDisplayRect,
		0, corners, true);

	// and maybe draw torus-wrapped ship(s)
	int xWrap =(playerx + SHIP_SPRITE_WIDTH) - SCREEN_WIDTH;
	int yWrap = (playery + SHIP_SPRITE_HEIGHT) - SCREEN_HEIGHT;
	int xShieldWrap = (playerx + SHIP_SPRITE_WIDTH/2 + shieldRadius) - SCREEN_WIDTH;
	int yShieldWrap = (playery + SHIP_SPRITE_HEIGHT/2 + shieldRadius) - SCREEN_HEIGHT;

	if(xWrap > 0 && yWrap > 0) {
		driver->draw2DImage(player->m_texture,
			core::rect<s32>(playerx-SCREEN_WIDTH,playery-SCREEN_HEIGHT,xWrap,yWrap),
			player->m_displayRect,
			0, 0, true);
	}
	if(xWrap > 0) {
		driver->draw2DImage(player->m_texture,
			core::rect<s32>(playerx-SCREEN_WIDTH,playery,xWrap,playery+SHIP_SPRITE_HEIGHT),
			player->m_displayRect,
			0, 0, true);
	}
	if(yWrap > 0) {
		driver->draw2DImage(player->m_texture,
			core::rect<s32>(playerx,playery-SCREEN_HEIGHT,playerx+SHIP_SPRITE_WIDTH,yWrap),
			player->m_displayRect,
			0, 0, true);
	}

	// and maybe draw torus-wrapped shield(s)
	if(xShieldWrap > 0 && yShieldWrap > 0) { // corner
		if (player->m_health < FLASH_HP_SLOW * MAX_SHIP_HEALTH) {
			driver->draw2DImage(whiteShield,
			core::rect<s32>((playerx+SHIP_SPRITE_WIDTH/2-shieldRadius) - SCREEN_WIDTH,
				(playery+SHIP_SPRITE_HEIGHT/2-shieldRadius) - SCREEN_HEIGHT,
				xShieldWrap,
				yShieldWrap),
			player->m_shieldDisplayRect,
			0, whiteCorners, true);
		}
		driver->draw2DImage(player->m_shield,
			core::rect<s32>((playerx+SHIP_SPRITE_WIDTH/2-shieldRadius) - SCREEN_WIDTH,
				(playery+SHIP_SPRITE_HEIGHT/2-shieldRadius) - SCREEN_HEIGHT,
				xShieldWrap,
				yShieldWrap),
			player->m_shieldDisplayRect,
			0, corners, true);
	}
	if(xShieldWrap > 0) { // sides
		if (player->m_health < FLASH_HP_SLOW * MAX_SHIP_HEALTH) {
			driver->draw2DImage(whiteShield,
			core::rect<s32>((playerx+SHIP_SPRITE_WIDTH/2-shieldRadius) - SCREEN_WIDTH,
				playery+SHIP_SPRITE_HEIGHT/2-shieldRadius,
				xShieldWrap,
				playery+SHIP_SPRITE_HEIGHT/2+shieldRadius),
			player->m_shieldDisplayRect,
			0, whiteCorners, true);
		}
		driver->draw2DImage(player->m_shield,
			core::rect<s32>((playerx+SHIP_SPRITE_WIDTH/2-shieldRadius) - SCREEN_WIDTH,
				playery+SHIP_SPRITE_HEIGHT/2-shieldRadius,
				xShieldWrap,
				playery+SHIP_SPRITE_HEIGHT/2+shieldRadius),
			player->m_shieldDisplayRect,
			0, corners, true);
	}
	if(yShieldWrap > 0) { // top/bottom
		if (player->m_health < FLASH_HP_SLOW * MAX_SHIP_HEALTH) {
			driver->draw2DImage(whiteShield,
			core::rect<s32>(playerx+SHIP_SPRITE_WIDTH/2-shieldRadius,
				(playery+SHIP_SPRITE_HEIGHT/2-shieldRadius) - SCREEN_HEIGHT,
				playerx+SHIP_SPRITE_WIDTH/2+shieldRadius,
				yShieldWrap),
			player->m_shieldDisplayRect,
			0, whiteCorners, true);
		}
		driver->draw2DImage(player->m_shield,
			core::rect<s32>(playerx+SHIP_SPRITE_WIDTH/2-shieldRadius,
				(playery+SHIP_SPRITE_HEIGHT/2-shieldRadius) - SCREEN_HEIGHT,
				playerx+SHIP_SPRITE_WIDTH/2+shieldRadius,
				yShieldWrap),
			player->m_shieldDisplayRect,
			0, corners, true);
	}
}

void drawMissile(Missile* missile) {
	int missilex = (int) (missile->m_position.X);
	int missiley = (int) (missile->m_position.Y);
	// draw normally
	driver->draw2DImage(missile->m_texture,
		core::rect<s32>(missilex,missiley,
		missilex + MISSILE_SPRITE_WIDTH, missiley + MISSILE_SPRITE_HEIGHT),
		missile->m_displayRect,
		0, 0, true);

	// and maybe draw torus-wrapped missiles(s)
	int xWrap =(missilex + MISSILE_SPRITE_WIDTH) - SCREEN_WIDTH;
	int yWrap = (missiley + MISSILE_SPRITE_HEIGHT) - SCREEN_HEIGHT;
	if(xWrap > 0 && yWrap > 0) {
		driver->draw2DImage(missile->m_texture,
			core::rect<s32>(missilex-SCREEN_WIDTH,missiley-SCREEN_HEIGHT,xWrap,yWrap),
			missile->m_displayRect,
			0, 0, true);
	}

	if(xWrap > 0) {
		driver->draw2DImage(missile->m_texture,
			core::rect<s32>(missilex-SCREEN_WIDTH,missiley,xWrap,missiley+MISSILE_SPRITE_HEIGHT),
			missile->m_displayRect,
			0, 0, true);
	}

	if(yWrap > 0) {
		driver->draw2DImage(missile->m_texture,
			core::rect<s32>(missilex,missiley-SCREEN_HEIGHT,missilex+MISSILE_SPRITE_WIDTH,yWrap),
			missile->m_displayRect,
			0, 0, true);
	}
}

void drawMine(Mine* mine) {
	int minex = (int) (mine->m_position.X);
	int miney = (int) (mine->m_position.Y);
	// draw normally
	driver->draw2DImage(mine->m_texture,
		core::rect<s32>(minex,miney,
		minex + MINE_SPRITE_WIDTH, miney + MINE_SPRITE_HEIGHT),
		mine->m_displayRect,
		0, 0, true);

	// and maybe draw torus-wrapped missiles(s)
	int xWrap =(minex + MINE_SPRITE_WIDTH) - SCREEN_WIDTH;
	int yWrap = (miney + MINE_SPRITE_HEIGHT) - SCREEN_HEIGHT;
	if(xWrap > 0 && yWrap > 0) {
		driver->draw2DImage(mine->m_texture,
			core::rect<s32>(minex-SCREEN_WIDTH,miney-SCREEN_HEIGHT,xWrap,yWrap),
			mine->m_displayRect,
			0, 0, true);
	}

	if(xWrap > 0) {
		driver->draw2DImage(mine->m_texture,
			core::rect<s32>(minex-SCREEN_WIDTH,miney,xWrap,miney+MINE_SPRITE_HEIGHT),
			mine->m_displayRect,
			0, 0, true);
	}

	if(yWrap > 0) {
		driver->draw2DImage(mine->m_texture,
			core::rect<s32>(minex,miney-SCREEN_HEIGHT,minex+MINE_SPRITE_WIDTH,yWrap),
			mine->m_displayRect,
			0, 0, true);
	}
}

bool drawHealthBar(Ship* player) {
	if (!player->m_exploded) {
		if (player->m_health > MAX_SHIP_HEALTH * FLASH_HP_SLOW) {
			return true;
		} else if (player->m_health > MAX_SHIP_HEALTH *  FLASH_HP_FAST) {
			if (flashCount%FLASH_TIME_SLOW > FLASH_TIME_SLOW/2) {

				return true;
			}
		} else if (flashCount%FLASH_TIME_FAST > FLASH_TIME_FAST/2) {
			return true;
		}
	}
	return false;
}

void newGame() {
	if (cfd)
		delete cfd;
	cfd = new CFDSolver(GRID_SIZE, 1, DIFFUSION, VISCOSITY);

	if (player1)
		delete player1;
	player1 = new Ship(cfd, player1Color, core::position2df(SCREEN_WIDTH/4.0-30.0, SCREEN_HEIGHT/2.0-10.0), 0, driver, 1, fmodSystem);
	
	if (player2)
		delete player2;
	player2 = new Ship(cfd, player2Color, core::position2df(SCREEN_WIDTH*3.0/4.0-30.0, SCREEN_HEIGHT/2.0-10.0), 180, driver, 2, fmodSystem);	

	timeNow = device->getTimer()->getTime();
	lastTime = timeNow-1;
	player1health = (int)MAX_SHIP_HEALTH;
	player2health = (int)MAX_SHIP_HEALTH;
	winner = 0;
	controlsUp = false;

	if (bgmChannel)
		errcheck(bgmChannel->setPaused(true));
	errcheck(fmodSystem->createStream("bgm/Sax_Quartet.mp3", FMOD_LOOP_NORMAL|FMOD_2D|FMOD_SOFTWARE, 0, &bgm));
	errcheck(fmodSystem->playSound(FMOD_CHANNEL_FREE, bgm, false, &bgmChannel));
}

void chooseColor(char color) {
	if((player1Color != COLOR_NONE && player2Color != COLOR_NONE) || !loaded)
		return;

	if(player1Color == COLOR_NONE) {
		switch(color) {
			case 'r': player1Color = COLOR_RED; player1IrrColor = video::SColor(255,255,0,0); break;
			case 'g': player1Color = COLOR_GREEN; player1IrrColor = video::SColor(255,0,255,0); break;
			case 'b': player1Color = COLOR_BLUE; player1IrrColor = video::SColor(255,0,0,255); break;
			default: break;
		}
	} else { // player2Color == COLOR_NONE
		switch(color) {
			case 'r': player2Color = COLOR_RED; player2IrrColor = video::SColor(255,255,0,0); break;
			case 'g': player2Color = COLOR_GREEN; player2IrrColor = video::SColor(255,0,255,0); break;
			case 'b': player2Color = COLOR_BLUE; player2IrrColor = video::SColor(255,0,0,255); break;
			default: break;
		}
		if(player2Color == player1Color)
			player2Color = COLOR_NONE;
		else {
			colorSelectUp = false;
			controlsUp = true;
		}
	}
}
