package arraylists;

import java.util.Arrays;

public class ArrList {
    String[] theArray;  // the underlying array that stores the elements
    int eltcount;       // how many elements are in the array
    int end;            // the last USED slot in the array
    /*
      We need eltcount to distinguish between a list with 0 and 1 elements.
      On its own, if end refers to last USED elt, a value of end=0 is ambiguous
         (does it mean list has 0 elements or 1?). Thus, we need a separate
         mechanism to track whether the list is empty. eltcount is one approach.
    */

    // the constructor
    public ArrList(int initSize) {
        this.end = 0;
        this.eltcount = 0;
        this.theArray = new String[initSize];
    }

    // this private method helps improve readability of later code
    private boolean isFull() {
        // theArray.length returns the CAPACITY of the array, not the # of filled slots
        return this.eltcount == this.theArray.length;
    }

    // this is a standard IList method (even though we haven't included the interface yet)
    public boolean isEmpty() {
        return this.eltcount == 0;
    }

    // get an element by its position in the list
    public String get (int index) {
        if ((index >= 0) && (index < this.theArray.length)) {
            return theArray[index];
        }
        throw new IllegalArgumentException("array index " + index + " out of bounds");
    }

    // this is the version of addLast that we had at the end of class.
    // it assumes there is enough space in the array for the newItem.
    // Java would raise an error if we add an element beyond the array's capacity
    public void addLastNoResize(String newItem) {
        if (!(isEmpty())) {
            // if array were empty, end already at 0, the right spot
            this.end = this.end + 1;
        }
        this.eltcount = this.eltcount + 1;
        this.theArray[this.end] = newItem;
    }

    // A method to make a new array to store list elements. This gets
    // called from a corrected addLast (see below) when the array is out of space
    private void resize(int newSize) {
        // make the new array
        String[] newArray = new String[newSize];
        // copy items from the current theArray to newArray
        for (int index = 0; index < theArray.length; index++) {
            newArray[index] = this.theArray[index];
        }
        // change this.theArray to refer to the new, larger array
        this.theArray = newArray;
    }

    // this is the corrected addLast, which handles additions beyond
    // the array capacity by first resizing the array, then adding the new element
    public void addLast(String newItem) {
        if (this.isFull()) {
            // add capacity to the array
            this.resize(this.theArray.length + 1);
            // now that the array has room, add the item
            this.addLast(newItem);
        } else {
            if (!(this.isEmpty())) {
                // only advance end if the list already had some element.
                // otherwise, list was empty and end should stay at 0
                this.end = this.end + 1;
            }
            // increase the element count (whether or not list was empty)
            this.eltcount = this.eltcount + 1;
            // store the new element
            this.theArray[end] = newItem;
        }
    }

    public String toString() {
        return Arrays.toString(this.theArray);
    }
}
