// Copyright (c) 2003 GeometryFactory
// All rights reserved.
//
// This file is part of CGAL (www.cgal.org); you may redistribute it under
// the terms of the Q Public License version 1.0.
// See the file LICENSE.QPL distributed with CGAL.
//
// Licensees holding a valid commercial license may use this file in
// accordance with the commercial license agreement provided with the software.
//
// This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
// $Source: /u/kungfu/cvs/KUNGFU/cswepson/src/ai_test/util/Interval_skip_list_interval.h,v $
// $Revision: 1.1 $ $Date: 2006/05/08 22:12:14 $
// $Name:  $
//
// Author(s)     : Andreas Fabri

#ifndef CGAL_INTERVAL_SKIP_LIST_INTERVAL_H
#define CGAL_INTERVAL_SKIP_LIST_INTERVAL_H

#include <iostream>

namespace util {

  template <class Value_, class Element_>
  class Interval_skip_list_interval
  {
  public:
    typedef Value_ Value;
	typedef Element_ Element;

  private:
    bool lbound_;
    bool rbound_;
    Value inf_;
    Value sup_;
	mutable Element elem_;

  public:

    Interval_skip_list_interval(){}
	Interval_skip_list_interval(const Interval_skip_list_interval& I);
    Interval_skip_list_interval(const Value& inf, const Value& sup,
				bool lb = true,	bool rb = true, 
				const Element& elem = Element());

	Element& element() const { return elem_; }

	void set_element(const Element& elem) { elem_ = elem; }

    const Value& inf() const {return inf_;}

    const Value& sup() const {return sup_;}

    bool inf_closed() const {return lbound_;}

    bool sup_closed() const {return rbound_;}

    bool contains(const Value& V) const;

    // true iff this contains (l,r)
    bool contains_interval(const Value& l, const Value& r) const;  

	const Interval_skip_list_interval& operator= (const Interval_skip_list_interval& I)
	{
		lbound_ = I.lbound_;
		rbound_ = I.rbound_;
		inf_ = I.inf_;
		sup_ = I.sup_;
		elem_ = I.elem_;
		return *this;
	}

    bool operator==(const Interval_skip_list_interval& I) const 
    {
      return ( (inf() == I.inf()) && (sup() == I.sup()) && 
	       (inf_closed() == I.inf_closed()) && (sup_closed() == I.sup_closed()) );
    }

    bool operator!=(const Interval_skip_list_interval& I) const 
    {
      return ! (*this == I);
    }
  };


template <class V, class E>
Interval_skip_list_interval<V,E>::Interval_skip_list_interval(const Interval_skip_list_interval<V,E>& I)
{
    lbound_ = I.lbound_;
    rbound_ = I.rbound_;
    inf_ = I.inf_;
    sup_ = I.sup_;
	elem_ = I.elem_;
}

  template <class V, class E>
  std::ostream& operator<<(std::ostream& os, 
			   const Interval_skip_list_interval<V, E>& i)
  {
    os << (i.inf_closed()?"[":"(") << i.inf() << ", " << i.sup() << (i.sup_closed()?"]":")");
    return os;
  }


  template <class V, class E>
  Interval_skip_list_interval<V, E>::Interval_skip_list_interval(
							      const Value& i, 
							      const Value& s,
								  bool lb, bool rb,
								  const Element& elem)
    : lbound_(lb), rbound_(rb), inf_(i), sup_(s), elem_(elem)
  {
    if ( inf_ > sup_ )
      {
	std::cerr << "Error: " << *this << std::endl;
	exit(1);
      }
  }


  template <class V, class E>
  bool
  Interval_skip_list_interval<V, E>::contains_interval(const Value& i, 
						    const Value& s) const
    // true iff this contains (l,r)
  {
    return( (inf() <= i) && (sup() >= s) );
  }


  template <class V, class E>
  bool
  Interval_skip_list_interval<V, E>::contains(const Value& v) const
  {
    // return true if this contains V, false otherwise
    if((v > inf()) && (v < sup()))
      return true;
    else if ((v == inf()) && inf_closed())
      return true;
    else if ((v == sup()) && sup_closed())
      return true;
    else
      return false;
  }

} // namespace os_util

#endif // CGAL_INTERVAL_SKIP_LIST_INTERVAL_H
