/////////////////////////////////////////////////////////////////////////////
// Name:        wx/univ/inpcons.h
// Purpose:     wxInputConsumer: mix-in class for input handling
// Author:      Vadim Zeitlin
// Modified by:
// Created:     14.08.00
// RCS-ID:      $Id: inpcons.h,v 1.2 2006/05/06 18:56:27 cswepson Exp $
// Copyright:   (c) 2000 SciTech Software, Inc. (www.scitechsoft.com)
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#ifndef _WX_UNIV_INPCONS_H_
#define _WX_UNIV_INPCONS_H_

#if defined(__GNUG__) && !defined(NO_GCC_PRAGMA)
    #pragma interface "inpcons.h"
#endif

class WXDLLEXPORT wxInputHandler;
class WXDLLEXPORT wxWindow;

#include "wx/object.h"
#include "wx/event.h"

// ----------------------------------------------------------------------------
// wxControlAction: the action is currently just a string which identifies it,
// later it might become an atom (i.e. an opaque handler to string).
// ----------------------------------------------------------------------------

typedef wxString wxControlAction;

// the list of actions which apply to all controls (other actions are defined
// in the controls headers)

#define wxACTION_NONE    _T("")           // no action to perform

// ----------------------------------------------------------------------------
// wxInputConsumer: mix-in class for handling wxControlActions (used by 
// wxControl and wxTopLevelWindow).
// ----------------------------------------------------------------------------

class WXDLLEXPORT wxInputConsumer
{
public:
    wxInputConsumer() { m_inputHandler = NULL; }

    // get the input handler
    wxInputHandler *GetInputHandler() const { return m_inputHandler; }

    // perform a control-dependent action: an action may have an optional
    // numeric and another (also optional) string argument whose interpretation
    // depends on the action
    //
    // NB: we might use ellipsis in PerformAction() declaration but this
    //     wouldn't be more efficient than always passing 2 unused parameters
    //     but would be more difficult. Another solution would be to have
    //     several overloaded versions but this will expose the problem of
    //     virtual function hiding we don't have here.
    virtual bool PerformAction(const wxControlAction& action,
                               long numArg = -1l,
                               const wxString& strArg = wxEmptyString);

    // get the window to work with (usually the class wxInputConsumer was mixed into)
    virtual wxWindow *GetInputWindow() const = 0;

protected:
    // event handlers
    void OnMouse(wxMouseEvent& event);
    void OnKeyDown(wxKeyEvent& event);
    void OnKeyUp(wxKeyEvent& event);
    void OnFocus(wxFocusEvent& event);
    void OnActivate(wxActivateEvent& event);

    // create input handler by name
    void CreateInputHandler(const wxString& inphandler);

    // input processor (never deleted, the theme deletes it itself)
    wxInputHandler *m_inputHandler;
};


// ----------------------------------------------------------------------------
// macros which must be used by the classes derived from wxInputConsumer mix-in
// ----------------------------------------------------------------------------

// declare the methods to be forwarded
#define WX_DECLARE_INPUT_CONSUMER() \
private: \
    void OnMouse(wxMouseEvent& event); \
    void OnKeyDown(wxKeyEvent& event); \
    void OnKeyUp(wxKeyEvent& event); \
    void OnFocus(wxFocusEvent& event); \
public: /* because of docview :-( */ \
    void OnActivate(wxActivateEvent& event); \
private:

// implement the event table entries for wxControlContainer
#define WX_EVENT_TABLE_INPUT_CONSUMER(classname) \
    EVT_KEY_DOWN(classname::OnKeyDown) \
    EVT_KEY_UP(classname::OnKeyUp) \
    EVT_MOUSE_EVENTS(classname::OnMouse) \
    EVT_SET_FOCUS(classname::OnFocus) \
    EVT_KILL_FOCUS(classname::OnFocus) \
    EVT_ACTIVATE(classname::OnActivate)
    
// Forward event handlers to wxInputConsumer
//
// (We can't use them directly, because wxIC has virtual methods, which forces 
// the compiler to include (at least) two vtables into wxControl, one for the 
// wxWindow-wxControlBase-wxControl branch and one for the wxIC mix-in. 
// Consequently, the "this" pointer has different value when in wxControl's 
// and wxIC's method, even though the instance stays same. This doesn't matter 
// so far as member pointers aren't used, but that's not wxControl's case. 
// When we add an event table entry (= use a member pointer) pointing to 
// wxIC's OnXXX method, GCC compiles code that executes wxIC::OnXXX with the 
// version of "this" that belongs to wxControl, not wxIC! In our particular 
// case, the effect is that m_handler is NULL (probably same memory
// area as the_other_vtable's_this->m_refObj) and input handling doesn't work.)
#define WX_FORWARD_TO_INPUT_CONSUMER(classname) \
    void classname::OnMouse(wxMouseEvent& event) \
    { \
        wxInputConsumer::OnMouse(event); \
    } \
    void classname::OnKeyDown(wxKeyEvent& event) \
    { \
        wxInputConsumer::OnKeyDown(event); \
    } \
    void classname::OnKeyUp(wxKeyEvent& event) \
    { \
        wxInputConsumer::OnKeyUp(event); \
    } \
    void classname::OnFocus(wxFocusEvent& event) \
    { \
        wxInputConsumer::OnFocus(event); \
    } \
    void classname::OnActivate(wxActivateEvent& event) \
    { \
        wxInputConsumer::OnActivate(event); \
    }

#endif // _WX_UNIV_INPCONS_H_
