/**
 @file odeHelper.h

  Routines that apply ODE functions to G3D types. 
  See also http://ode.org/ode-latest-userguide.htm

 Dojo Game Engine

 Copyright 2005, Morgan McGuire
 All rights reserved.
 */
#ifndef ODEHELPER_H
#define ODEHELPER_H

#include "g3dHelper.h"
#include <ode/ode.h>

/** Used when creating objects that are not in a collision space. */
#define ODE_NO_SPACE 0

void dGeomSetPositionAndRotation(dGeomID id, const CoordinateFrame& cframe);
void dGeomGetPositionAndRotation(dGeomID id, CoordinateFrame& cframe);

void dGeomGetPosition(dGeomID id, Vector3& pos);
void dGeomSetPosition(dGeomID id, const Vector3& pos);
void dGeomTranslate(dGeomID id, const Vector3& wsDelta);

void dBodySetPositionAndRotation(dBodyID id, const CoordinateFrame& cframe);

void dBodySetLinearVel(dBodyID id, const Vector3& v);
void dBodySetAngularVel(dBodyID id, const Vector3& v);

void dBodyGetLinearVel(dBodyID id, Vector3& v);
void dBodyGetAngularVel(dBodyID id, Vector3& v);
void dBodyGetLinearAndAngularVel(dBodyID id, Vector3& L, Vector3& A);

void dBodyGetPositionAndRotation(dBodyID id, CoordinateFrame& c);

void dJointSetHingeAnchor(dJointID id, const Vector3& v);
void dJointSetHingeAxis(dJointID id, const Vector3& v);
void dJointGetHingeAnchor(dJointID ID, Vector3& anchor);
void dJointGetHingeAxis(dJointID ID, Vector3& axis);

void dJointSetUniversalAnchor(dJointID id, const Vector3& v);
void dJointSetUniversalAxes(dJointID id, const Vector3& v, const Vector3& v2);
void dJointGetUniversalAxes(dJointID ID, Vector3& axis, Vector3& axis2);

void dJointGetBallAnchor(dJointID ID, Vector3& anchor);

/** Returns true if these geoms are in the same body or bodies that are connected by a joint. 
    Useful in collision detection.*/
int dAreConnected(dGeomID g1, dGeomID g2);

int dAreConnectedExcluding(dGeomID g1, dGeomID g2, int joint_type);

/** Applies this coordinate frame's world to object transformation. */
void dMassToObjectSpace(dMass *, const CoordinateFrame& cframe);

/** Applies this coordinate frame's object to world transformation. */
void dMassToWorldSpace(dMass *, const CoordinateFrame& cframe);

void dMassTranslate (dMass *, const Vector3& t);
void dMassRotate (dMass *, const Matrix3& M);

inline G3D::int32 hashCode(const dGeomID& g) {
    return reinterpret_cast<G3D::int32>(g);
}

inline G3D::int32 hashCode(const dBodyID& g) {
    return reinterpret_cast<G3D::int32>(g);
}

inline G3D::int32 hashCode(const dJointID& g) {
    return reinterpret_cast<G3D::int32>(g);
}

#endif
