/**
 @file HUDUI.h

 Dojo Game Engine

 Copyright 2005, Morgan McGuire
 All rights reserved.
 */
#ifndef DOJO_HUDUI_H
#define DOJO_HUDUI_H

#include "g3dHelper.h"

namespace dojo {

/** 
 Only used inside DApp.
 */
class HUDUI {
private:
    TextureRef          playIcon;
    TextureRef          pauseIcon;
    TextureRef          stepIcon;

    GFontRef            smallFont;
    GFontRef            labelFont;

    /**
    Assumes push2D mode. 
    */
    void drawTimeBreakdown(const Vector2& pos, float radius) const;

    void drawPhysicsDebugger(const Vector2& pos) const;

    class DApp*         app;

    class Window {
    public:
        enum Type {WINDOW_NONE, WINDOW_PHYSICS, WINDOW_GRAPHICS};

        HUDUI*          hudUI;

        std::string     title;
        Rect2D          rect;
        bool            visible;
        bool            resizable;

    protected:

        Window(HUDUI* h) : hudUI(h) {}

    public:
        virtual ~Window() {}

        /** Default draws the window. */
        virtual void draw(bool inFocus);

        virtual Type type() const {
            return WINDOW_NONE;
        }

        /** Create a specific window type using new.  
            Delete the result with delete.
            */
        static Window* fromType(Type t, HUDUI* h);
    };

    class PhysicsDebugger : public Window {
    public:

        /** Called by Window::fromType() */
        PhysicsDebugger(HUDUI* h);

        void draw(bool inFocus);

        virtual Type type() const {
            return WINDOW_PHYSICS;
        }
    };

    class GraphicsDebugger : public Window {
    public:
        
        /** Called by Window::fromType() */
        GraphicsDebugger(HUDUI* h);

        void draw(bool inFocus);

        virtual Type type() const {
            return WINDOW_GRAPHICS;
        }
    };

    /** Drawn in this order. If a NULL is on top of the stack then no window is active.*/
    Array<Window*>      windowStack;

    /**
     Returns the index in windowStack of the window with this type. 
     Returns -1 if there is no window of that type.*/
    int windowIndex(Window::Type t) const;

    Rect2D              timeWindow;

    /** True if we are currently dragging the window 
        on top of the stack. */
    bool                inDrag;

    /** Distance from the mouse to the top left corner of the active window during a drag. */
    Vector2             dragOffset;

    /** True if mouse down should be processed (i.e. we are not 
        waiting on a button click or drag to complete) */
    bool                processMouseDown;

public:

    HUDUI();

    /** Called by DApp's constructor */
    void loadData(DApp* app, GFontRef smallFont);

    /** Called by DApplet */
    void onGraphics(RenderDevice* rd);

    /** Called by DApplet */
    void onUserInput(UserInput* ui);
};
}

#endif
